/* This is an implementation of the RMDP class. 

   File: rmdp.cpp
   Las modified: May, 2002.
   Copyright: Bohdana Ratitch, SOCS, McGill University
*/

#include <stdlib.h>
#include <string.h>
#include<stdio.h>
#include<math.h>
#include<sys/types.h>		
#include<iostream.h>
#include<time.h>

#ifndef INTERFACE_CLASSES	
	#define INTERFACE_CLASSES
	#include "interface_classes.h"
#endif

#ifndef PWC_CLASS	
	#define PWC_CLASS
	#include "pwc.h"
#endif

#ifndef RMDP_CLASS	
	#define RMDP_CLASS
	#include "rmdp.h"
#endif

#define generate_constraints 3 //private use only

RMDP::RMDP(int task, RMDPparameters& p, ActionSet** as, bool save)
  /* This constructor generates an RMDP with parameters passed with p argument.
  */
{
  char* fileName=NULL;
  char* buffer=NULL;
  char* command=NULL;
  int i;

  if ((save==true)||(task==load)){

    if (p.rmdpName==NULL){
      cout << "Must specify the name of the RMDP" << endl;
      exit(EXIT_FAILURE);
    }
    
    if (p.rmdpName[strlen(p.rmdpName)-1]==DIR_SEP[0])
      p.rmdpName[strlen(p.rmdpName)-1]='\0';

    if (p.rmdpName[strlen(p.rmdpName)-1]=='.'){
      cout << ". not a valid name for an RMDP" << endl;
      exit(EXIT_FAILURE);
    }
 
  
    //create the directory 
    if ((task==generate) || (task==generate_with_constraints)){
      command=new char[10+strlen(p.rmdpName)];
      strcpy(command,"mkdir -p ");
      strcat(command,p.rmdpName);
      if (system(command)==-1){
	cout << "Cannot create or find directory " << p.rmdpName << endl;
	exit(EXIT_FAILURE);
      }
    }
  
    //get the directory name without full path and form file name without extension
  
    buffer=strrchr(p.rmdpName, DIR_SEP[0]);
    if(buffer==NULL){
      fileName = new char[2*strlen(p.rmdpName)+6];
      strcpy(fileName, p.rmdpName);
      strcat(fileName,DIR_SEP);
      strcat(fileName, p.rmdpName);
    }
    else {
      fileName = new char[strlen(p.rmdpName)+strlen(buffer)+6];
      strcpy(fileName, p.rmdpName);
      strcat(fileName,DIR_SEP);
      strcat(fileName, buffer+1);
    }
  }

  
  if(task==load){
    
    strcat(fileName,".rgp");
    loadRMDP(fileName, as);
    //record loaded parameters into p structure
    if (Type==0) p.Type='c';
    else p.Type='d';
    p.Variables=Variables;
    p.Actions=Actions;
    p.Discretize=new int[Variables];
    for (i=0; i<Variables; i++)
      p.Discretize[i]=Discretize[i];
    p.BranchingFactor=new int[Actions];
    for (i=0; i<Actions; i++)
      p.BranchingFactor[i]=BranchingFactor[i];

  }

  if (task==generate)
    generateRMDP(p, as);

  if (task==generate_with_constraints){
    
    if (p.UDC!=NULL){
      generateRMDP(p.UDC, as);
    }
    else{
      cout << "Error: specify the name of the file with user defined constraints" << endl;
      exit(EXIT_FAILURE);
    }
  }

  if (save==true)
    saveRMDP(p.rmdpName, (*as));

  delete [] fileName;
  delete [] command;
}

void RMDP::loadRMDP(char* rmdpFileName, ActionSet** as)
	/*	Loads a previously generated (or user created) MDP from a description a text file fileName
		Returns a new action set to the calling function.
	*/
{
  char* buffer = new char[150];
  int branch, variable, action;
  char c;
  ifstream rmdpFile(rmdpFileName);
  if (rmdpFile.fail()){
    cout << "Cannot open file " << rmdpFileName << " for reading" << endl;
    exit(EXIT_FAILURE);
  }

  
  rmdpFile.get(buffer, strlen("Type ")+1);
  rmdpFile.get(c);
  if (c=='d') Type=1;
  else Type=0;
  rmdpFile.get(c);//get eol
  rmdpFile.get(buffer, strlen("Variables ")+1);
  rmdpFile >> Variables;
  State::dimensionality=Variables;
  rmdpFile.get(c); //get endl;

  rmdpFile.get(buffer, strlen("Actions ")+1);
  rmdpFile >> Actions;
  rmdpFile.get(c); //get endl;
  //create an action set

  (*as) = new ActionSet(Actions);
  for (action=0; action<Actions; action++){
    sprintf(buffer, "%d", action);
    Action a(buffer, action);
    (*as)->addAction(a);
  }

  maxBF=0;
  BranchingFactor = new int[Actions];

  rmdpFile.get(buffer, strlen("BranchingFactor ")+1);
  for(action=0;action<Actions;action++){
    rmdpFile >> BranchingFactor[action];
     rmdpFile.get(c);//get " " or eol
    if (BranchingFactor[action]>maxBF) maxBF=BranchingFactor[action];
  }
  
  if (rmdpFile.fail()){
    cout << "Error reading from file (check)" << rmdpFileName << endl;
    exit(EXIT_FAILURE);
  }

  char*** Name = new char**[Actions];
  for (action=0; action<Actions;action++){
    Name[action]=new char*[maxBF];
    for(branch=0;branch<maxBF;branch++)
      Name[action][branch]= new char[150];
  }
  
  Discretize = new int[Variables];
  //State::NumberOfValues = new int[Variables];
  rmdpFile.get(buffer,strlen("Discretize ")+1);
  for (variable=0; variable<Variables; variable++){
    rmdpFile >> Discretize[variable];
    rmdpFile.get(c);//get ' ' or eol
  }
  
  if (rmdpFile.fail()){
    cout << "Error reading from file " << rmdpFileName << endl;
    exit(EXIT_FAILURE);
  }
  
  rmdpFile.get(buffer, strlen("Branching Probability functions:")+1);
  rmdpFile.get(c); //get endl;
  if (strncmp(buffer, "Branching Probability functions:", strlen("Branching Probability functions:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }

  char* path = new char[150];
  char* name = new char[150];
  char* point;
  point=strrchr(rmdpFileName,DIR_SEP[0]);
  if (point==NULL){
    strcpy(path,".");
    strcat(path,DIR_SEP);
  }
  else{
    strncpy(path,rmdpFileName,(point+1-rmdpFileName));
    path[(point+1-rmdpFileName)]='\0';
  }
	
  Approximator** tempFA;
	
  BranchingProbability = new StateActionFA*[maxBF];
  for (action=0; action<Actions; action++){	
    for(branch=0; branch<maxBF; branch++){
      if (branch<BranchingFactor[action]){
	rmdpFile.get(c); //get ' '
	rmdpFile >> name;	
	if (rmdpFile.fail()){
	  cout << "Error reading file " << rmdpFileName;
	  exit(EXIT_FAILURE);
	}
	strcpy(Name[action][branch],path);
	strcat(Name[action][branch],name);
      }
      else Name[action][branch]=NULL;
    }
    rmdpFile.get(c); //get endl;
  }

  for(branch=0; branch<maxBF; branch++){
    tempFA = new Approximator*[Actions];
    for (action=0; action<Actions; action++){	
      if (branch<BranchingFactor[action]){
	tempFA[action] = new pwc();
	tempFA[action]->setArchitectureParameters(1, &(Name[action][branch]));
      }
      else tempFA[action]=NULL;
    }
    BranchingProbability[branch] = new StateActionFA(Actions, tempFA);
  }

  rmdpFile.get(buffer, strlen("Transition Mean functions:")+1);
  if (strncmp(buffer, "Transition Mean functions:", strlen("Transition Mean functions:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile.get(c); //get endl;

  TransitionMean = new StateActionFA**[Variables];
  for(variable=0; variable<Variables; variable++){
    for (action=0; action<Actions; action++){	
      for(branch=0; branch<maxBF; branch++){
	if (branch<BranchingFactor[action]){
	  rmdpFile.get(c); //get ' '
	  rmdpFile >> name;	
	  if (rmdpFile.fail()){
	    cout << "Error reading file " << rmdpFileName;
	    exit(EXIT_FAILURE);
	  }
	  strcpy(Name[action][branch],path);
	  strcat(Name[action][branch],name);
	}
	else Name[action][branch]=NULL;
      }
      rmdpFile.get(c); //get endl;
    }
	  
    TransitionMean[variable] = new StateActionFA*[maxBF];
    for(branch=0; branch<maxBF; branch++){
      tempFA = new Approximator*[Actions];
      for (action=0; action<Actions; action++){
	if (branch<BranchingFactor[action]){
	  tempFA[action]= new pwc();
	  tempFA[action]->setArchitectureParameters(1,&(Name[action][branch]));
	}
	else tempFA[action]=NULL;
      }
      TransitionMean[variable][branch] = new StateActionFA(Actions, tempFA);
    }
  }
	
  rmdpFile.get(buffer, strlen("Transition Variance functions:")+1);
  if (strncmp(buffer, "Transition Variance functions:", strlen("Transition Variance functions:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile.get(c); //get endl;
  
	
  TransitionVariance = new StateActionFA**[Variables];
  for(variable=0; variable<Variables; variable++){
    for (action=0; action<Actions; action++){	
      for(branch=0; branch<maxBF; branch++){
	if (branch<BranchingFactor[action]){
	  rmdpFile.get(c); //get ' '
	  rmdpFile >> name;	
	  if (rmdpFile.fail()){
	    cout << "Error reading file " << rmdpFileName;
	    exit(EXIT_FAILURE);
	  }
	  strcpy(Name[action][branch],path);
	  strcat(Name[action][branch],name);
	}
	else Name[action][branch]=NULL;
      }
      rmdpFile.get(c); //get endl;
    }

    TransitionVariance[variable] = new StateActionFA*[maxBF];
    for(branch=0; branch<maxBF; branch++){
      tempFA = new Approximator*[Actions];
      for (action=0; action<Actions; action++){
	if (branch<BranchingFactor[action]){
	  tempFA[action] = new pwc();
	  tempFA[action]->setArchitectureParameters(1, &(Name[action][branch]));
	}
	else tempFA[action]=NULL;
	TransitionVariance[variable][branch] = new StateActionFA(Actions, tempFA);
      }
    }
  }
    
  rmdpFile.get(buffer, strlen("Reward Mean functions:")+1);
  if (strncmp(buffer, "Reward Mean functions:", strlen("Reward Mean functions:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile.get(c); //get endl;
  tempFA = new Approximator*[Actions];
  for (action=0; action<Actions; action++){
    tempFA[action] = new pwc();
    rmdpFile.get(c);//get" " 
    rmdpFile >> name;
    if (rmdpFile.fail()){
      cout << "Error reading file " << rmdpFileName;
      exit(EXIT_FAILURE);
    }
    strcpy(Name[0][0],path);
    strcat(Name[0][0],name);
    
    tempFA[action]->setArchitectureParameters(1, &(Name[0][0]));
  }
  RewardMean  = new StateActionFA(Actions, tempFA);
  rmdpFile.get(c); //get endl;
  
  rmdpFile.get(buffer, strlen("Reward Variance functions:")+1);
  if (strncmp(buffer, "Reward Variance functions:", strlen("Reward Variance functions:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile.get(c); //get endl;
  tempFA = new Approximator*[Actions];
  for (action=0; action<Actions; action++){
    tempFA[action] = new pwc();
    rmdpFile.get(c); //get " " 
    rmdpFile >> name;
    if (rmdpFile.fail()){
      cout << "Error reading file " << rmdpFileName;
      exit(EXIT_FAILURE);
    }
    strcpy(Name[0][0],path);
    strcat(Name[0][0],name);

    tempFA[action]->setArchitectureParameters(1, &(Name[0][0]));
  }
  RewardVariance = new StateActionFA(Actions, tempFA);
  rmdpFile.get(c); //get endl;

  rmdpFile.get(buffer, strlen("Termination Probability function:")+1);
  if (strncmp(buffer, "Termination Probability function:", strlen("Termination Probability function:"))!=0){
    cout << "Formatting error in file " << rmdpFileName << endl;
    cout << "Reading " << buffer << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile.get(c); //get endl;
  TerminationProbability = new pwc();
  rmdpFile.get(c); //get " " 
  rmdpFile >> name;
  if (rmdpFile.fail()){
    cout << "Error reading file " << rmdpFileName;
    exit(EXIT_FAILURE);
  }
  strcpy(Name[0][0],path);
  strcat(Name[0][0],name);
  
  TerminationProbability->setArchitectureParameters(1, &(Name[0][0]));
}

void RMDP::saveRMDP(char* dirName, ActionSet* as)
  /*	Saves parameters of the generated RMDP into a text file with the name fileName.
   */
{
  char* buffer = new char[150];
  char* rmdpFileName;
  char* baseName;
  char** Name = new char*[1];
  Name[0]=new char[250];
  int branch, variable, action;
  char* path;
  char* command;

  if (dirName[strlen(dirName)-1]==DIR_SEP[0])
    dirName[strlen(dirName)-1]='\0';

  if (dirName[strlen(dirName)-1]=='.'){
    cout << ". is not a valid name for an RMDP" << endl;
    exit(EXIT_FAILURE);
  }

  //create the directory to save the data
  command=new char[10+strlen(dirName)];
  strcpy(command,"mkdir -p ");
  strcat(command,dirName);
  if (system(command)==-1){
    cout << "Cannot create or find directory " << dirName << endl;
    exit(EXIT_FAILURE);
  }
  
  //get the directory name without full path and form base file name without path and extention
  buffer=strrchr(dirName, DIR_SEP[0]);
  if(buffer==NULL){
    baseName=new char[strlen(dirName)+1];
    strcpy(baseName,dirName);
    rmdpFileName = new char[2*strlen(dirName)+10];
  }
  else {
    baseName=new char[strlen(buffer)];
    strcpy(baseName,buffer+1);
    rmdpFileName = new char[strlen(dirName)+strlen(buffer)+10];
  }

  strcpy(rmdpFileName, dirName);
  strcat(rmdpFileName,DIR_SEP);
  strcat(rmdpFileName, baseName);
  strcat(rmdpFileName, ".rgp");
  path=new char[strlen(dirName)+2];
  strcpy(path, dirName);
  strcat(path, DIR_SEP);

  //start saving

  ofstream rmdpFile(rmdpFileName);
  if (rmdpFile.fail()){
    cout << "Cannot open file " << rmdpFileName << " for writing" << endl;
    exit(EXIT_FAILURE);
  }
  rmdpFile << "Type ";
  if (Type==0)
    rmdpFile << "c" << endl;
  else rmdpFile << "d" << endl;
  rmdpFile << "Variables " << Variables << endl;
  rmdpFile << "Actions " << Actions << endl;
  rmdpFile << "BranchingFactor" ;
  for (action=0;action<Actions;action++)
    rmdpFile << " " << BranchingFactor[action];
  rmdpFile << endl;

  rmdpFile << "Discretize ";
  for(variable=0; variable<Variables; variable++)
    rmdpFile << " " << Discretize[variable];
  rmdpFile << endl;
  if (rmdpFile.fail()){
    cout << "Error writing to file " << rmdpFileName << endl;
    exit(EXIT_FAILURE);
  }

  rmdpFile << "Branching Probability functions:" << endl;
  if (rmdpFile.fail()){
    cout << "Error writing to file " << rmdpFileName << endl;
    exit(EXIT_FAILURE);
  }
  char* tmpName = new char[200];
  for (action=0; action<Actions; action++){
    for(branch=0; branch<BranchingFactor[action]; branch++){
      sprintf(Name[0], "%s%s%d%s%d%s", baseName, "-b", branch, "-a", action, ".bpf");
      rmdpFile << " " << Name[0];
      strcpy(tmpName,Name[0]);
      strcpy(Name[0],path);
      strcat(Name[0],tmpName);
      BranchingProbability[branch]->saveArchitectureParameters(as->action[action], 1, Name);
	    
      if (rmdpFile.fail()){
	cout << "Error writing to file " << rmdpFileName << endl;
	exit(EXIT_FAILURE);
      }
    }
    rmdpFile << endl;
  }
	
  rmdpFile<< "Transition Mean functions:" << endl;
  for(variable=0; variable<Variables; variable++){
    for (action=0; action<Actions; action++){
      for(branch=0; branch<BranchingFactor[action]; branch++){
			
	sprintf(Name[0], "%s%s%d%s%d%s%d%s", baseName, "-v", variable, "-b", branch, "-a", action, ".tmf");
	rmdpFile << " " << Name[0];
	strcpy(tmpName,Name[0]);
	strcpy(Name[0],path);
	strcat(Name[0],tmpName);
	TransitionMean[variable][branch]->saveArchitectureParameters(as->action[action],1,Name);
				
      }
      rmdpFile << endl;
    }
  }

  rmdpFile<< "Transition Variance functions:" << endl;
  for(variable=0; variable<Variables; variable++){
    for (action=0; action<Actions; action++){
      for(branch=0; branch<BranchingFactor[action]; branch++){
			
	sprintf(Name[0], "%s%s%d%s%d%s%d%s", baseName, "-v", variable, "-b", branch, "-a", action, ".tvf");
	rmdpFile << " " << Name[0];
	strcpy(tmpName,Name[0]);
	strcpy(Name[0],path);
	strcat(Name[0],tmpName);
	TransitionVariance[variable][branch]->saveArchitectureParameters(as->action[action],1,Name);
				
      }
      rmdpFile << endl;
    }
  }

  rmdpFile << "Reward Mean functions:" << endl;
  for (action=0; action<Actions; action++){
    sprintf(Name[0], "%s%s%d%s", baseName, "-a", action, ".rmf");
    rmdpFile << " " << Name[0];
    strcpy(tmpName,Name[0]);
    strcpy(Name[0],path);
    strcat(Name[0],tmpName);
    RewardMean->saveArchitectureParameters(as->action[action],1,Name);
			
  }
  rmdpFile << endl;

  rmdpFile << "Reward Variance functions:" << endl;
  for (action=0; action<Actions; action++){
    sprintf(Name[0], "%s%s%d%s", baseName, "-a", action, ".rvf");
    rmdpFile << " " << Name[0];
    strcpy(tmpName,Name[0]);
    strcpy(Name[0],path);
    strcat(Name[0],tmpName);
    RewardVariance->saveArchitectureParameters(as->action[action],1,Name);
			
  }
  rmdpFile << endl;

  rmdpFile << "Termination Probability function:" << endl;
  sprintf(Name[0], "%s%s", baseName, ".tpf");
  rmdpFile << " " << Name[0];
  strcpy(tmpName,Name[0]);
  strcpy(Name[0],path);
  strcat(Name[0],tmpName);
  TerminationProbability->saveArchitectureParameters(1, Name);
	
}

void RMDP::generateRMDP(char* fileName, ActionSet** as)
  /*	Generates a new random MDP using constraints specified in the text file fileName
	Returns a new action set to the calling function.
  */
{
  char* buffer = new char[150];
  char c;
  int i, j, k;
  int temp;
  int action, variable, branch; //indeces for arrays TM and TV
  double low, high, variance, prob;

  ifstream inFile(fileName);
  if (inFile.fail()){
    cout << "Cannot open file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  inFile.get(buffer, strlen("Type ")+1);
  inFile.get(c);
  if (c=='d') Type=1;
  else Type=0;
  inFile.get(c);//get eol;
	
  inFile.get(buffer, strlen("Variables ")+1);
  inFile >> State::dimensionality;
  inFile.get(c);
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  if (State::dimensionality < 1){
    cout << "Number of state variables must be at least 1" << endl;
    exit(EXIT_FAILURE);
  }

  Variables = State::dimensionality;

  inFile.get(buffer, strlen("Actions ")+1);
  inFile >> temp;
  inFile.get(c);
	
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  if (temp < 1){
    cout << "Number of actions must be at least 1" << endl;
    exit(EXIT_FAILURE);
  }

  //form a new action set

  (*as) = new ActionSet(temp);
  for (i=0; i<temp; i++){
    sprintf(buffer, "%d", i);
    Action a(buffer, i);
    (*as)->addAction(a);
  }

  Actions=temp;
	
  BranchingFactor = new int[Actions];
	
  inFile.get(buffer, strlen("BranchingFactor ")+1);
  for(i=0;i<Actions;i++){
    inFile >> BranchingFactor[i];
    inFile.get(c);
    if (inFile.fail()){
      cout << "Error reading from file " << fileName << endl;
      exit(EXIT_FAILURE);
    }
    if (BranchingFactor[i] < 1){
      cout << "Branching factor must be at least 1" << endl;
      exit(EXIT_FAILURE);
    }
  }
	
  inFile.get(buffer, strlen("DependencyFactor ")+1);
  inFile >> DependencyFactor;
  inFile.get(c);

  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  if ((DependencyFactor < 1) || (DependencyFactor>Variables)){
    cout << "Dependency factor must be no less than 1 and no greater than number of variables" << endl;
    exit(EXIT_FAILURE);
  }

	
  //read into how many intervals each dimension should be discretized

  int* intervals = new int[State::dimensionality];
  int** depend = new int*[State::dimensionality];
  for (i=0; i<State::dimensionality; i++)
    depend[i]= new int[State::dimensionality];
	
  inFile.get(buffer, strlen("Discretize ")+1);
  for (i=0; i<State::dimensionality; i++){
    inFile >> intervals[i];
    inFile.get(c);
    if (intervals[i]<=1){
      cout << "Each dimension should be descritized into more than one intervals" << endl;
      exit(EXIT_FAILURE);
    }
  }
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  Discretize = new int[State::dimensionality];
  for(i=0;i<State::dimensionality;i++)
    Discretize[i]=intervals[i];

	
  //randomly generate dependencies among variables
  //depend is an array that indicates if variables depend on each other
  //if depend[i][j]==1 the var. i does not depend on var j, otherwise it does.

  int ind;
  for (i=0; i<State::dimensionality; i++){
    for (j=0; j<State::dimensionality; j++)
      depend[i][j]=intervals[j];
    for(j=0; j<(State::dimensionality-DependencyFactor); j++){
      ind=rand()%State::dimensionality;
      while (depend[i][ind]==1){
	ind++;
	if (ind>=State::dimensionality) ind=0;
      }
      depend[i][ind]=1;
    }
  }

  //prepare variables bound for pwc's
  double* left = new double[State::dimensionality];
  double* right = new double[State::dimensionality];

  for(i=0; i<State::dimensionality; i++){
    left[i]=0;
    right[i]=1;
  }

  Approximator** tempSAFA;
  Subset s1, s2;
  int BF=0;
  for(i=0;i<Actions;i++)
    if (BranchingFactor[i]>BF) BF=BranchingFactor[i];

  maxBF=BF;

  //Create data structures for functions representing parameters 
  //for transitions, rewards and termination.

  //Transition means and variances

  BranchingProbability = new StateActionFA*[BF];
  TransitionMean = new StateActionFA**[State::dimensionality];
  TransitionVariance = new StateActionFA**[State::dimensionality];
  for (i=0; i< State::dimensionality; i++){
    TransitionMean[i] = new StateActionFA*[BF];
    TransitionVariance[i] = new StateActionFA*[BF];
  }


	
  pwc**** TM = new pwc***[BF];
  pwc**** TV = new pwc***[BF];

  for (i=0; i<BF; i++){
    TM[i] = new pwc**[State::dimensionality];
    TV[i] = new pwc**[State::dimensionality];
    for (j=0; j<State::dimensionality; j++){
      TM[i][j] = new pwc*[(*as)->size];
      TV[i][j] = new pwc*[(*as)->size];
    }
  }

  pwc*** BP = new pwc**[BF];
  for (i=0; i<BF; i++)
    BP[i] = new pwc*[(*as)->size];


  double* origin = new double[State::dimensionality];
  double* orPtr;
  if (Type==0) orPtr=NULL;
  else{
    orPtr = new double[2*State::dimensionality];
    
    for(variable=0; variable<2*Variables; variable++)
      orPtr[variable]=0.0;
  }
	
  inFile.get(buffer, strlen("Transitions:")+1);
  inFile.get(c);//get eol
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }
  
  inFile.get(c); //reading 'a'
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  //get path to the .str file
  char* path =  new char[250];
  char* bff;
  bff=strrchr(fileName,DIR_SEP[0]);
  if (bff==NULL){
    strcpy(path,".");
    strcat(path, DIR_SEP);
  }
  else{
    strncpy(path,fileName,(bff+1-fileName));
    path[(bff+1-fileName)]='\0';
  }
	

  for (action=0; action<(*as)->size; action++){
			
    if (c!='a'){
      cout << "Formatting error in file: transitions, action spec.(1) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile.get(c); //get ' '
    inFile >> temp; //number of the action
    inFile.get(c);
    if (inFile.fail()){
      cout << "Error reading from file " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    if (temp!=action){
      cout << "Formatting error in file: transitions, action spec.(2) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    //read default constraints
    inFile.get(buffer, strlen("f d")+1);
    inFile.get(c);
    if (inFile.fail()){
      cout << "Error reading from file " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    if (strncmp(buffer, "f d", 3)!=0){
      cout << "Formatting error in file: transitions, default constraints (1) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    for (branch=0; branch<BF; branch++){
      if (branch>=BranchingFactor[action]){
	for(variable=0; variable<State::dimensionality; variable++){
	  TM[branch][variable][action] = NULL;
	  TV[branch][variable][action] = NULL;
	}
	BP[branch][action] = NULL;
	continue;
      }
		  
      inFile.get(buffer, strlen("t ")+1);
      if (strncmp(buffer, "t ", 2)!=0){
	cout << "Formatting error in file : transitions, default constraints (2)" << fileName << endl;
	exit(EXIT_FAILURE);
      }
			
      //read transition means
      for(variable=0; variable<State::dimensionality; variable++){
	inFile.get(c); //get [
	if (c!='['){
	  cout << "Formatting error in file : transitions, default constraints (3)" << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile >> low;
	inFile.get(c); //get ,
	inFile >> high;
	inFile.get(c); //get ]
	if (c!=']'){
	  cout << "Formatting error in file : transitions, default constraints (3/4)" << fileName << endl;
	  cout << "reading " << c << " character" << endl;
	  exit(EXIT_FAILURE);
	}
	inFile.get(c); //get ' '

	TM[branch][variable][action] = new pwc(Variables, left, right, depend[variable], low, high, orPtr, Discretize[variable]*Type);

      }

      //read transition variances

      for(variable=0; variable<State::dimensionality; variable++){
	inFile.get(c); //get (
	if (c!='('){
	  cout << "Formatting error in file : transitions, default constraints (3/4(v))" << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile >> low;
	inFile.get(c); //get ,
	inFile >> high;
	inFile.get(c); //get )
	if (c!=')'){
	  cout << "Formatting error in file : transitions, default constraints (3/4(v))" << fileName << endl;
	  cout << "reading " << c << " character" << endl;
	  exit(EXIT_FAILURE);
	}
	inFile.get(c); //get ' '

	TV[branch][variable][action] = new pwc(Variables, left, right, depend[variable], low, high, orPtr);
      }
			

      //read branching probabilities
      inFile.get(c);
      if (c!='p'){
	cout << "Formatting error in file: transitions, default constraints (5)" << fileName << endl;
	exit(EXIT_FAILURE);
      }
      inFile.get(c); //get =
      inFile >> prob;
      inFile.get(c); //get eol
      if ((prob<0) || (prob>1)){
	cout << "Branching probability incorrect" << endl;
	exit(EXIT_FAILURE);
      }

      if (branch==0){
	BP[branch][action] = new pwc(Variables, left, right, intervals, prob, prob, orPtr);
	BP[branch][action]->getOrigin(origin);
      }
      else
	BP[branch][action] = new pwc(Variables, left, right, intervals, prob, prob, origin);

    }	//end of branch loop
	
    if (inFile.fail()){
      cout << "Error reading from file " << fileName << endl;
      exit(EXIT_FAILURE);
    }

		

    //read other non-default constrains for state space subsets
    inFile.get(c);
    while (c=='f'){

      inFile.get(c); //get ' '
			

      for(variable=0; variable<State::dimensionality; variable++){
	inFile.get(c); //get [
	if (c!='['){
	  cout << "Formatting error in file: transitions, non-default constraints (1) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}

	inFile >> s1.left[variable];
	inFile.get(c); //get ,
	inFile >> s1.right[variable];
	inFile.get(c); //get ]
	inFile.get(c); //get ' ' or eol
      }

      for (branch=0; branch<BF; branch++){
	if (branch>=BranchingFactor[action]) continue;
	inFile.get(buffer, strlen("t ")+1);
	if (strncmp(buffer, "t ", 2)!=0){
	  cout << "Formatting error in file: transitions, non-default constraints (2) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
			
				//read transition means
	for(variable=0; variable<State::dimensionality; variable++){
	  inFile.get(c); //get [
	  if (c!='['){
	    cout << "Formatting error in file: transitions, non-default constraints (3) " << fileName << endl;
	    exit(EXIT_FAILURE);
	  }
	  inFile >> low;
	  inFile.get(c); //get ,
	  inFile >> high;
	  inFile.get(c); //get ]
	  inFile.get(c); //get ' '

	  TM[branch][variable][action]->weightInterval(s1, low, high, Discretize[variable]*Type);
	}

				//read transition variances

	for(variable=0; variable<State::dimensionality; variable++){
	  inFile.get(c); //get (
	  if (c!='('){
	    cout << "Formatting error in file: transitions, non-default constraints (3(v)) " << fileName << endl;
	    exit(EXIT_FAILURE);
	  }
	  inFile >> low;
	  inFile.get(c); //get ,
	  inFile >> high;
	  inFile.get(c); //get )
	  inFile.get(c); //get ' '

	  TV[branch][variable][action]->weightInterval(s1, low, high);
	}

				//read branching probabilities
	inFile.get(c);
	if (c!='p'){
	  cout << "Formatting error in file: transitions, non-default constraints (5) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile.get(c); //get =
	inFile >> prob;
	inFile.get(c); //get eol
	if ((prob<0) || (prob>1)){
	  cout << "Branching probability incorrect" << endl;
	  exit(EXIT_FAILURE);
	}

	BP[branch][action]->weightInterval(s1, prob, prob);

      }	//end of branch loop for non-default constraints
			
      inFile.get(c); //get first char in a new line

    } //end of while loop for non-default constraints
		
  } //end of action loop for transitions

  //form StateAction architectures for transitions

  //	cout << "Transition constraints read successfully" << endl;
		
  for (branch=0; branch<BF; branch++){
    tempSAFA = new Approximator*[(*as)->size];
    for (action=0; action < (*as)->size; action++)
      tempSAFA[action] = BP[branch][action];
    BranchingProbability[branch] = new StateActionFA((*as)->size, tempSAFA);
    for (variable=0; variable<State::dimensionality; variable++){
      tempSAFA = new Approximator*[(*as)->size];
      for (action=0; action < (*as)->size; action++)
	tempSAFA[action] = TM[branch][variable][action];
      TransitionMean[variable][branch]= new StateActionFA((*as)->size, tempSAFA);
      tempSAFA = new Approximator*[(*as)->size];
      for (action=0; action < (*as)->size; action++)
	tempSAFA[action] = TV[branch][variable][action];
      TransitionVariance[variable][branch] = new StateActionFA((*as)->size, tempSAFA);
    }
  }

  //read constraints for rewards

  //pwcDS** RM = new pwcDS*[(*as)->size];
  //pwcDS** RV = new pwcDS*[(*as)->size];

  pwc** RM = new pwc*[(*as)->size];
  pwc** RV = new pwc*[(*as)->size];
  double* leftDS=new double[Variables*2];
  double* rightDS=new double[Variables*2];
  for (i=0; i<Variables*2; i++){
    leftDS[i]=0.0;
    rightDS[i]=1.0;
  }
  Subset s1DS(2);
  int* intervalsDS = new int[Variables*2];
  for(i=0; i<Variables; i++)
    intervalsDS[i]=intervals[i];
  for(i=Variables; i<2*Variables; i++)
    intervalsDS[i]=intervals[i-Variables];

  //character "R" has been read in the previous loop
	
  inFile.get(buffer, strlen("ewards: ")+1);
  if (inFile.fail()){
    cout << "Error reading from file " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  if ((c!='R')||(strncmp(buffer, "ewards:", 7)!=0)){
    cout << "Formatting error in file: rewards " << fileName << endl;
    exit(EXIT_FAILURE);
  }

  inFile.get(c); //get eol
  inFile.get(c); //get a

  for(action=0; action<(*as)->size; action++){
		
    if (c!='a'){
      cout << "Formatting error in file: rewards, action spec. (1)" << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile.get(c); //get ' '
    inFile >> temp;
    inFile.get(c); //get eol
    if (temp!=action){
      cout << "Formatting error in file: rewards, action spec. (2) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    //read default constraint

    inFile.get(buffer, strlen("f d")+1);
    if (strncmp(buffer, "f d", 3)!=0){
      cout << "Formatting error in file: reward, default constraints (1) " << fileName << endl;
      exit(EXIT_FAILURE);
    }
    inFile.get(c); //get eol

    inFile.get(c); //get t
    if (c!='t'){
      cout << "Formatting error in file: reward, default constraints (2) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile.get(c); //get ' '
    inFile.get(c); //get [
    if (c!='['){
      cout << "Formatting error in file: reward, default constraints (3) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile >> low;
    inFile.get(c); //get ,
    if (c!=','){
      cout << "Formatting error in file: reward, default constraints (4) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile >> high;
    inFile.get(c); //get ]
    if (c!=']'){
      cout << "Formatting error in file: reward, default constraints (5) " << fileName << endl;
      exit(EXIT_FAILURE);
    }
    //here
    inFile.get(c); //get ' '
    RM[action] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, low, high, orPtr);

    //read reward variance
    inFile.get(c); //get (
    if (c!='('){
      cout << "Formatting error in file: reward, default constraints (3) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile >> low;
    inFile.get(c); //get ,
    if (c!=','){
      cout << "Formatting error in file: reward, default constraints (4) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

    inFile >> high;
    inFile.get(c); //get )
    if (c!=')'){
      cout << "Formatting error in file: reward, default constraints (5) " << fileName << endl;
      exit(EXIT_FAILURE);
    }

		
    inFile.get(c); //get eol
    RV[action] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, low, high, orPtr);

    //read non-default constraints
		
    inFile.get(c); //get f
    while(c=='f'){
      inFile.get(c); //get ' ' 
			
      for(variable=0; variable<State::dimensionality; variable++){
	inFile.get(c); //get [
	if (c!='['){
	  cout << "Formatting error in file: reward, non-default constraints (1) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile >> s1DS.left[variable];
	inFile.get(c); //get ,
				
	inFile >> s1DS.right[variable];
	inFile.get(c); //get ]
	if (c!=']'){
	  cout << "Formatting error in file: reward, non-default constraints (2) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile.get(c); //get ' '
      }

      inFile.get(buffer, strlen("and ")+1);
      if (strncmp(buffer, "and ", 4)!=0){
	cout << "Formatting error in file: reward, non-default constraints (3) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      for(variable=0; variable<State::dimensionality; variable++){
					
	inFile.get(c); //get [
	if (c!='['){
	  cout << "Formatting error in file: reward, non-default constraints (4) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
				
	inFile >> s1DS.left[Variables+variable];
	inFile.get(c); //get ,
	inFile >> s1DS.right[Variables+variable];
	inFile.get(c); //get ]
	if (c!=']'){
	  cout << "Formatting error in file: reward, non-default constraints (5) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile.get(c); //get ' ' or eol
      }

      inFile.get(c); //get t
      if (c!='t'){
	cout << "Formatting error in file: reward, non-default constraints (5) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile.get(c); //get ' '
      inFile.get(c); //get [
      if (c!='['){
	cout << "Formatting error in file: reward, non-default constraints (6) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile >> low;
      inFile.get(c); //get ,
      if (c!=','){
	cout << "Formatting error in file: reward, non-default constraints (7) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile >> high;
      inFile.get(c); //get ]
      if (c!=']'){
	cout << "Formatting error in file: reward, non-default constraints (8) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile.get(c); //get ' '
      RM[action]->weightInterval(s1DS, low,high);

      //get variance
      inFile.get(c); //get (
      if (c!='('){
	cout << "Formatting error in file: reward, non-default constraints (6) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile >> low;
      inFile.get(c); //get ,
      if (c!=','){
	cout << "Formatting error in file: reward, non-default constraints (7) " << fileName << endl;
	exit(EXIT_FAILURE);
      }

      inFile >> high;
      inFile.get(c); //get )
      if (c!=')'){
	cout << "Formatting error in file: reward, non-default constraints (8) " << fileName << endl;
	exit(EXIT_FAILURE);
      }
			
      inFile.get(c); //get eol
      RV[action]->weightInterval(s1DS, low, high);
			
      inFile.get(c); //get f 
			
    } //end of while loop for non-default reward constraints

  }//end of action loop for rewards

  //cout << "Reward constraints read successfully" << endl;
	
  tempSAFA = new Approximator*[(*as)->size];
  for(action=0; action < (*as)->size; action++)
    tempSAFA[action] = RM[action];
  RewardMean = new StateActionFA((*as)->size, tempSAFA);
  tempSAFA = new Approximator*[(*as)->size];
  for(action=0; action < (*as)->size; action++)
    tempSAFA[action] = RV[action];
  RewardVariance = new StateActionFA((*as)->size, tempSAFA);

  //read constraints for termination probabilities
	
  //"T" has been already read in the previous loop
  inFile.get(buffer, strlen("ermination probability:")+1);
  inFile.get(c); //get eol
	
  //read default constraint
  inFile.get(c); //get d
  if (c!='d'){
    cout << "Formatting error in file: termination probabilities, default (1)" << fileName << endl;
    cout << "Reading " << c << " charracter " << endl; 
    exit(EXIT_FAILURE);
  }

  inFile.get(c); //get ' ' 
  inFile.get(c); //get p
  inFile.get(c); //get =
  if (c!='='){
    cout << "Formatting error in file: termination probabilities, default (2) " << fileName << endl;
    exit(EXIT_FAILURE);
  }
  inFile >> prob;
  inFile.get(c); //get eol

  pwc* TP = new pwc(Variables, left, right, intervals, prob, prob, orPtr);

  //read non-default constraints
  inFile.get(c); //get f
  if (inFile.eof()==0){//not the end of file
    while(c=='f'){
      inFile.get(c); //get ' ' 
      for (variable=0; variable<State::dimensionality; variable++){
	inFile.get(c); //get [
	if (c!='['){
	  cout << "Formatting error in file: termination probabilities, non-default (1) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile >> s1.left[variable];
	inFile.get(c); //get ,
	if (c!=','){
	  cout << "Formatting error in file: termination probabilities, non-default (2) " << fileName << endl;
	  exit(EXIT_FAILURE);
	}
	inFile >> s1.right[variable];
	inFile.get(c); //get ]
	inFile.get(c); //get ' '
      }

      inFile.get(c); //get p
      inFile.get(c); //get =
      if (c!='='){
	cout << "Formatting error in file: termination probabilities, non-default (3) " << fileName << endl;
	exit(EXIT_FAILURE);
      }
      inFile >> prob;

      TP->weightInterval(s1, prob, prob);

      inFile.get(c); //get eol
      inFile.get(c); //get first char on the next line
      if (inFile.eof()!=0) break;
    }//end of while loop for termination probabilities		
  } //if not eof
	
  TerminationProbability = TP;

  //	cout << "Termination probability constraints read successfully" << endl;

  //memory cleanup
  delete [] buffer;
  delete [] intervals;

  for (i=0; i<State::dimensionality; i++)
    delete [] depend[i];
  delete [] depend;

  delete [] left;
  delete [] right;

  for(branch=0; branch<BF; branch++){
    for(variable=0; variable<State::dimensionality; variable++){
      delete [] TM[branch][variable];
      delete [] TV[branch][variable];
    }
    delete [] TM[branch];
    delete [] TV[branch];
  }
  delete [] TM;
  delete [] TV;

  for(branch=0; branch<BF; branch++)
    delete [] BP[branch];
  delete [] BP;

  delete [] RM;
  delete [] RV;
		
}

/////////
void RMDP::generateRMDP(RMDPparameters& p, ActionSet** as)
  /*	Generates a new random MDP.
  */
{
  p.check(generate);

  int i, j, N;
  int action, variable, branch; //indeces for arrays TM and TV
  double prob;
  bool UseMaxBF=false;
  int CurrentBF=1;
  char* buffer = new char[5];

  if (p.Type=='d') Type=1;
  else Type=0;
	
  State::dimensionality=p.Variables;
  Variables = p.Variables;

  (*as) = new ActionSet(p.Actions);
  for (i=0; i<p.Actions; i++){
    sprintf(buffer, "%d", i);
    Action a(buffer, i);
    (*as)->addAction(a);
  }

  Actions=p.Actions;
	
  BranchingFactor = new int[Actions];
	
  if (p.BranchingFactor!=NULL){
    for(i=0;i<Actions;i++){
      BranchingFactor[i]=p.BranchingFactor[i];
      if (BranchingFactor[i] < 1){
	cout << "Branching factor must be at least 1" << endl;
	exit(EXIT_FAILURE);
      }
    }
  }
  else{
    if (p.MaxBranchingFactor!=NULL){
      UseMaxBF=true;
      for(i=0;i<Actions;i++){
	BranchingFactor[i]=p.MaxBranchingFactor[i];
	if (BranchingFactor[i] < 1){
	  cout << "Branching factor must be at least 1" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }
    else{
      cout << "Branching Factors not specified." << endl;
      exit(EXIT_FAILURE);
    }
  }

  maxBF=0;
  for(i=0;i<Actions;i++)
    if (BranchingFactor[i]>maxBF) maxBF=BranchingFactor[i];
  
  DependencyFactor=p.DependencyFactor;

  //read into how many intervals each dimension should be discretized

  int* intervals = new int[State::dimensionality];
  int** depend = new int*[State::dimensionality];
  for (i=0; i<State::dimensionality; i++)
    depend[i]= new int[State::dimensionality];
	
  for (i=0; i<State::dimensionality; i++){
    intervals[i]=p.Discretize[i];
    if (intervals[i]<=1){
      cout << "Each dimension should be descritized into more than one intervals" << endl;
      exit(EXIT_FAILURE);
    }
  }

  Discretize = new int[State::dimensionality];
  for(i=0;i<State::dimensionality;i++)
    Discretize[i]=intervals[i];

	
  //randomly generate dependencies among variables
  //depend is an array that indicates if variables depend on each other
  //if depend[i][j]==1 the var. i does not depend on var j, otherwise it does.

  int ind;
  for (i=0; i<State::dimensionality; i++){
    for (j=0; j<State::dimensionality; j++)
      depend[i][j]=intervals[j];
    for(j=0; j<(State::dimensionality-DependencyFactor); j++){
      ind=rand()%State::dimensionality;
      while (depend[i][ind]==1){
	ind++;
	if (ind>=State::dimensionality) ind=0;
      }
      depend[i][ind]=1;
    }
  }

  //prepare variables bound for pwc's
  double* left = new double[State::dimensionality];
  double* right = new double[State::dimensionality];
  double* leftDS = new double[State::dimensionality*2];
  double* rightDS = new double[State::dimensionality*2];

  for(i=0; i<State::dimensionality; i++){
    left[i]=0;
    right[i]=1;
  }

  for(i=0; i<State::dimensionality*2; i++){
    leftDS[i]=0;
    rightDS[i]=1;
  }
  
  //Create data structures for functions representing parameters 
  //for transitions, rewards and termination.

  Approximator** tempSAFA;

  //Transition means and variances

  BranchingProbability = new StateActionFA*[maxBF];
  TransitionMean = new StateActionFA**[State::dimensionality];
  TransitionVariance = new StateActionFA**[State::dimensionality];
  for (i=0; i< State::dimensionality; i++){
    TransitionMean[i] = new StateActionFA*[maxBF];
    TransitionVariance[i] = new StateActionFA*[maxBF];
  }


	
  pwc**** TM = new pwc***[maxBF];
  pwc**** TV = new pwc***[maxBF];

  for (i=0; i<maxBF; i++){
    TM[i] = new pwc**[State::dimensionality];
    TV[i] = new pwc**[State::dimensionality];
    for (j=0; j<State::dimensionality; j++){
      TM[i][j] = new pwc*[(*as)->size];
      TV[i][j] = new pwc*[(*as)->size];
    }
  }

  pwc*** BP = new pwc**[maxBF];
  for (i=0; i<maxBF; i++)
    BP[i] = new pwc*[(*as)->size];

  
  if (p.TransitionSimilarity==false){
    p.BPsimilarity=rand()%101;
    p.TMsimilarity=rand()%101;
    p.TVsimilarity=rand()%101;
  }
  if (p.RewardSimilarity==false){
    p.RMsimilarity=rand()%101;
    p.RVsimilarity=rand()%101;
  }
  
  //generate transition functions
  
  //generate for the first action
  action=0;
  double* origin = new double[Variables];
  double* orPtr;
  if (Type==0) orPtr=NULL;
  else{
    orPtr = new double[2*Variables];
    for (variable=0; variable<2*Variables; variable++)
      orPtr[variable]=0.0;
  }
  
  for (branch=0; branch<maxBF; branch++){
    if (branch>=BranchingFactor[action]){
      for(variable=0; variable<State::dimensionality; variable++){
	TM[branch][variable][action] = NULL;
	TV[branch][variable][action] = NULL;
      }
      BP[branch][action] = NULL;
    }
    else{
      for(variable=0; variable<State::dimensionality; variable++){
	TM[branch][variable][action] = new pwc(Variables, left, right, depend[variable], 0, 1, orPtr, Discretize[variable]*Type);
	TV[branch][variable][action] = new pwc(Variables, left, right, depend[variable], p.minTV[variable][action], p.maxTV[variable][action], orPtr);
      }
      if (branch==0){
	BP[branch][action] = new pwc(Variables, left, right, intervals, 0, 1, orPtr);
	BP[branch][action]->getOrigin(origin);
      }
      else{
	BP[branch][action] = new pwc(Variables, left, right, intervals, 0, 1, origin);
      }
    }
  }	//end of branch loop

  N=BP[0][0]->getSize();
  
  //make sure branching probabilities sum to 1
  for (i=0; i<N; i++){
    if (UseMaxBF==true) 
      CurrentBF=(rand()%BranchingFactor[action])+1;
    else 
      CurrentBF=BranchingFactor[action];
    prob=0;
    for(branch=0; branch<BranchingFactor[action]; branch++){
      if (branch<CurrentBF)
	prob+=BP[branch][action]->getParameter(i);
      else
	BP[branch][action]->setParameter(i,0.0);
    }
    if (prob==0) 
      BP[0][action]->setParameter(i, 1.0);
    else{
      for(branch=0; branch<BranchingFactor[action]; branch++)
	BP[branch][action]->setParameter(i, BP[branch][action]->getParameter(i)/prob);
    }
  }


        
  //generate transitions for other actions taking uinto account "similarity" requirement
  int* dare = new int[N];
  double* w = new double[N];
 
  

  BP[0][0]->getOrigin(origin); //get origin of the branching prop. functions. Must be the same for all branches and actions
  
  for (action=1; action<Actions; action++){
    

    if (BranchingFactor[action]==BranchingFactor[0]){//see if branching probabilities will be purturbed
      for(i=0; i<N;i++)
	if ((rand()%101) > p.BPsimilarity) dare[i]=1;
	else dare[i]=0;
    }

    for (branch=0; branch<maxBF; branch++){
      if (branch>=BranchingFactor[action]){
	for(variable=0; variable<State::dimensionality; variable++){
	  TM[branch][variable][action] = NULL;
	  TV[branch][variable][action] = NULL;
	}
	BP[branch][action] = NULL;
      }
      else{
	if (BranchingFactor[action]==BranchingFactor[0]){//purturb 0's actions architecture

	  BP[branch][0]->getParameters(w);
	  purturb(N, w, 0, 0, 0.0, 1.0, dare);
	  BP[branch][action] = new pwc(Variables, left, right, intervals, 0, 1, origin);
	  BP[branch][action]->setParameters(w);
	}
	else{//branching factors not equal - generate as usual
	  BP[branch][action] = new pwc(Variables, left, right, intervals, 0, 1, origin);
	}
	if (branch<BranchingFactor[0]){//purturb 0th action's architectures on this branch
	  if (orPtr==NULL) orPtr = new double[2*Variables];
	  for (variable=0; variable<Variables; variable++){

	    TM[branch][variable][0]->getParameters(w);
	    purturb(N, w, p.TMsimilarity, Discretize[variable]*Type);
	    TM[branch][variable][0]->getOrigin(orPtr);
	    TM[branch][variable][action] = new pwc(Variables, left, right, depend[variable], 0, 1, orPtr, Discretize[variable]*Type);
	    TM[branch][variable][action]->setParameters(w);
	   
	    TV[branch][variable][0]->getParameters(w);
	    purturb(N, w, p.TVsimilarity, 0, p.minTV[variable][action], p.maxTV[variable][action]);
	    TV[branch][variable][0]->getOrigin(orPtr);
	    TV[branch][variable][action] = new pwc(Variables, left, right, depend[variable], 0 , 1, orPtr);
	    TV[branch][variable][action]->setParameters(w);
	  }//end of variable loop
	}
	else{//generate normally
	  delete [] orPtr;
	  if (Type==0) orPtr=NULL;
	  else{
	    orPtr = new double[2*Variables];
	    for (variable=0; variable<2*Variables; variable++)
	      orPtr[variable]=0.0;
	  }
	  for(variable=0; variable<State::dimensionality; variable++){
	    TM[branch][variable][action] = new pwc(Variables, left, right, depend[variable], 0, 1, orPtr, Discretize[variable]*Type);
	    TV[branch][variable][action] = new pwc(Variables,left, right, depend[variable], p.minTV[variable][action], p.maxTV[variable][action], orPtr);
	  }//end of variable loop
	}//else - generate, not purturb
      }//else - branch exists for this action
  }//end of branch loop

    //make sure probabilities sum to 1
    for (i=0; i<N; i++){
      if (UseMaxBF==true) 
	CurrentBF=(rand()% BranchingFactor[action])+1;
      else 
	CurrentBF=BranchingFactor[action];
      prob=0;
      for(branch=0; branch<BranchingFactor[action]; branch++){
	if (branch<CurrentBF)
	  prob+=BP[branch][action]->getParameter(i);
	else
	  BP[branch][action]->setParameter(i,0.0);
      }
      if (prob==0) 
	BP[0][action]->setParameter(i, 1.0);
      else{
	for(branch=0; branch<BranchingFactor[action]; branch++)
	  BP[branch][action]->setParameter(i, BP[branch][action]->getParameter(i)/prob);
      }
    }
  }//end of action loop

  //form StateAction architectures for transitions

  for (branch=0; branch<maxBF; branch++){
    tempSAFA = new Approximator*[(*as)->size];
    for (action=0; action < (*as)->size; action++)
      tempSAFA[action] = BP[branch][action];
    BranchingProbability[branch] = new StateActionFA((*as)->size, tempSAFA);
    for (variable=0; variable<State::dimensionality; variable++){
      tempSAFA = new Approximator*[(*as)->size];
      for (action=0; action < (*as)->size; action++)
	tempSAFA[action] = TM[branch][variable][action];
      TransitionMean[variable][branch]= new StateActionFA((*as)->size, tempSAFA);
      tempSAFA = new Approximator*[(*as)->size];
      for (action=0; action < (*as)->size; action++)
	tempSAFA[action] = TV[branch][variable][action];
      TransitionVariance[variable][branch] = new StateActionFA((*as)->size, tempSAFA);
    }
  }


  //generate reward functions

  //pwcDS** RM = new pwcDS*[(*as)->size];
  //pwcDS** RV = new pwcDS*[(*as)->size];
  
  pwc** RM = new pwc*[(*as)->size];
  pwc** RV = new pwc*[(*as)->size];
  int* intervalsDS = new int[Variables*2];
  for(i=0; i<Variables; i++)
    intervalsDS[i]=intervals[i];
  for(i=Variables; i<2*Variables; i++)
    intervalsDS[i]=intervals[i-Variables];

  delete [] orPtr;
  if (Type==0) orPtr=NULL;
  else{
    orPtr = new double[2*Variables];
    for (variable=0; variable<2*Variables; variable++)
      orPtr[variable]=0.0;
  }
  
  RM[0] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, 0, 1, orPtr);
  RV[0] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, p.minRV[0], p.maxRV[0], orPtr);
  N=RM[0]->getSize();
  
  delete [] w;
  w= new double[N];
  if (orPtr==NULL) orPtr = new double[2*Variables];

  for(action=1; action<(*as)->size; action++){
    
    //purturb reward means and variances
    RM[0]->getParameters(w);
    purturb(N, w, p.RMsimilarity);
    RM[0]->getOrigin(orPtr);
    RM[action] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, 0, 1, orPtr);
    RM[action]->setParameters(w);

    RV[0]->getParameters(w);
    purturb(N, w, p.RVsimilarity, 0, p.minRV[action], p.maxRV[action]);
    RV[0]->getOrigin(orPtr);
    RV[action] = new pwc(Variables*2, leftDS, rightDS, intervalsDS, 0, 1, orPtr);
    RV[action]->setParameters(w);
    
  }
	
  tempSAFA = new Approximator*[(*as)->size];
  for(action=0; action < (*as)->size; action++)
    tempSAFA[action] = RM[action];
  RewardMean = new StateActionFA((*as)->size, tempSAFA);
  tempSAFA = new Approximator*[(*as)->size];
  for(action=0; action < (*as)->size; action++)
    tempSAFA[action] = RV[action];
  RewardVariance = new StateActionFA((*as)->size, tempSAFA);

  //generate termination probabilities
 

  pwc* TP = new pwc(Variables, left, right, intervals, p.minTP, p.maxTP, orPtr);
  TerminationProbability = TP;


  //memory cleanup
  delete [] dare;
  delete [] w;
  delete [] intervals;
  delete [] orPtr;

  for (i=0; i<State::dimensionality; i++)
    delete [] depend[i];
  delete [] depend;

  delete [] left;
  delete [] right;

  for(branch=0; branch<maxBF; branch++){
    for(variable=0; variable<State::dimensionality; variable++){
      delete [] TM[branch][variable];
      delete [] TV[branch][variable];
    }
    delete [] TM[branch];
    delete [] TV[branch];
  }
  delete [] TM;
  delete [] TV;
}
/////////

void RMDP::uniformStateSample(State& s){
  int i;

  if (Type==0){
    for (i=0; i<State::dimensionality; i++)
      s.x[i]=ran1(&idum);
  }
  else{
    for (i=0; i<State::dimensionality; i++)
      s.x[i]=(double)(rand()%Discretize[i])/(double)(Discretize[i])+1.0/(double)(2*Discretize[i]);
  }
}

void RMDP::startState(State& start, bool& terminal)
{
  int i;
  double prob;
	
  if (Type==0){
    for (i=0; i<State::dimensionality; i++)
      start.x[i]=ran1(&idum);
  }
  else{
    for (i=0; i<State::dimensionality; i++)
      start.x[i]=(double)(rand()%Discretize[i])/(double)(Discretize[i])+1.0/(double)(2*Discretize[i]);
  }
		
  CurrentState=start;

  //compute termination probability in this state
  TerminationProbability->predict(start, prob);
  if (ran1(&idum)<prob)
    terminal=true;
  else terminal=false;
}


bool RMDP::applicable(const State& s, const Action& a)
  /*	Checks if action a is applicable in state s.
	*/
{
  if ((a.value<0) || (a.value>Actions)) return false;
  else return true;
}


void RMDP::bound(int i, bool& bounded, double& left, double& right)
  /*	Gives bounds on state variables' values
		Parameters:
			i : index of state variable
			bounded: indicates if i^th variable is bounded
			left : left bound
			right: right bound
	*/
{
  bounded = true;
  left=0;
  right=1;
}

	
void RMDP::setState(const State& s, bool& terminal)
  /* Sets the current state to s and records if the state is terminal
	*/
{
  double prob;
  CurrentState = s;

  //compute termination probability in this state
  TerminationProbability->predict(s, prob);
  if (ran1(&idum)<=prob)
    terminal=true;
  else terminal=false;
}

	
void RMDP::transition(const Action& action, State& s_new, double& r_new, bool& terminal)
  /*	Implements a transtion in responce to the action 
		performed by the agent. Updates its internal variables 
		and delivers values to the agent.
		Parameters:
			action: action performed by the agent
			s_new : return value - new state
			r_new : return value - new reward
			terminal: indication of whether s_new is a terminal state
	*/
{
  double prob;
  double* BP = new double[maxBF];
  double mean, variance;
  int TakenBranch;
  int i;


  BranchingProbability[0]->predict(action, CurrentState, BP[0]);
  for (i=1; i<BranchingFactor[action.id]; i++){
    BranchingProbability[i]->predict(action, CurrentState, BP[i]);
    BP[i]=BP[i-1] + BP[i];
  }
	
  //select a branch
  prob = ran1(&idum);
  i=0;
  while (prob>BP[i]){
    i++;
    if (i>=BranchingFactor[action.id]){
      i=BranchingFactor[action.id]-1;
      break;
    }
  }
  TakenBranch=i;
  //sample new state
  for(i=0; i<State::dimensionality; i++){
    TransitionMean[i][TakenBranch]->predict(action, CurrentState, mean);
    TransitionVariance[i][TakenBranch]->predict(action, CurrentState, variance);
    s_new.x[i]=sqrt(variance)*gasdev(&idum)+mean;
    if (s_new.x[i]<0) s_new.x[i]=0;
    if (s_new.x[i]>1) s_new.x[i]=1;
  }

  //sample new reward
  State current_new_state(2);
  for(i=0; i<State::dimensionality; i++){
    current_new_state.x[i]=CurrentState.x[i];
    current_new_state.x[i+State::dimensionality]=s_new.x[i];
  }
  RewardMean->predict(action, current_new_state, mean);
  RewardVariance->predict(action, current_new_state,variance);
  r_new = sqrt(variance)*gasdev(&idum)+mean;
  if (r_new<0) r_new=0;
  if (r_new>1) r_new=1;

	
  //compute termination probability in this state
  TerminationProbability->predict(s_new, prob);
  if (ran1(&idum)<prob)
    terminal=true;
  else terminal=false;

  CurrentState=s_new;
  CurrentAction=action;

  delete [] BP;
}
	
RMDP::~RMDP()
{	
  int i, j;
  delete RewardMean;
  delete RewardVariance;

  for (i=0; i<maxBF; i++)
    delete BranchingProbability[i];

  delete [] BranchingProbability;
  delete TerminationProbability;

  for (i=0; i<State::dimensionality; i++){
    for (j=0; j<maxBF; j++){
      delete TransitionMean[i][j];
      delete TransitionVariance[i][j];
    }
    delete [] TransitionMean[i];
    delete [] TransitionVariance[i];
  }
  delete [] TransitionMean;
  delete [] TransitionVariance;
  delete [] BranchingFactor;
  delete [] Discretize;
	
}

void RMDP::generateRMDPconstraints(RMDPparameters& p)
{
  p.check(generate_constraints);
 
  char* buffer = new char[150];
  char c;
  int i;
  char* outFileName=new char[strlen(p.UDC)+1];
  
  strcpy(outFileName, p.UDC);
  
  ofstream outFile(outFileName);
  if(outFile.fail()){
    cout << "Error: cannot open file " << outFileName << endl;
    exit(EXIT_FAILURE);
  }

  outFile << "Type " << p.Type << endl;
  outFile << "Variables " << p.Variables << endl;
  outFile << "Actions " << p.Actions << endl;
  outFile << "BranchingFactor";
  if (p.BranchingFactor!=NULL){
    for(i=0; i<p.Actions; i++){
      outFile << " " << p.BranchingFactor[i];
    }
  }
  else{
    for(i=0; i<p.Actions; i++){
      outFile << " " << p.MaxBranchingFactor[i];
    }
  }
  outFile << endl;
  outFile << "DependencyFactor " << p.DependencyFactor << endl;
  outFile << "Discretize";
  for(i=0; i<p.Variables; i++){
    outFile << " " << p.Discretize[i];
  }
  outFile << endl;
 
  outFile << "Transitions:" << endl;
  
  int variable, action, bf, rule;
  double* BP = NULL;
  double BPsum;
  int CurrentBF, CurrentMaxBF;
  double left, right, tmp, length;
  double variance;

  for (action=0; action<p.Actions; action++){

    delete [] BP;
    if (p.MaxBranchingFactor==NULL)
      BP = new double[p.BranchingFactor[action]];
    else
      BP = new double[p.MaxBranchingFactor[action]];
	  
    outFile << "a " << action << endl;
    //generate default constraints
    outFile << "f d" << endl;
		
    //generate branching probabilities
    BPsum=0;
    if (p.BranchingFactor==NULL){
      CurrentBF = rand()%p.MaxBranchingFactor[action]+1;
      CurrentMaxBF=p.MaxBranchingFactor[action];
    }
    else{
      CurrentBF=p.BranchingFactor[action];
      CurrentMaxBF=p.BranchingFactor[action];
    }
		
    for(bf=0; bf<CurrentMaxBF; bf++){
      if (bf<CurrentBF){
	BP[bf]=(double)rand()/(double)RAND_MAX;
	BPsum+=BP[bf];
      }
      else
	BP[bf]=0;
    }
    if (BPsum==0){
      BP[0]=1;
      BPsum=1;
    }

    for(bf=0; bf<CurrentMaxBF; bf++){
      outFile << "t ";
			
      for (variable=0; variable<p.Variables; variable++){
	outFile << "[" << 0 << "," << 1 << "] "; 
      }
      //write variance
			
      for (variable=0; variable<p.Variables; variable++){
	outFile << "(" << p.minTV[variable][action] << "," << p.maxTV[variable][action] << ") "; 
      }
      outFile << "p=" << BP[bf]/BPsum << endl;	
      if (outFile.fail()){
	cout << "Error writing to file " << outFileName << endl;
	exit(EXIT_FAILURE);
      }
    }

    //generate non-default rules
    for (rule=0; rule<p.TransitionRules[action]; rule++){
      outFile << "f ";
      for(variable=0; variable<p.Variables; variable++){

	length = (double)rand()/(double)RAND_MAX*p.TIL[variable][action];
	left = (double)rand()/(double)RAND_MAX*(1-length);
	right = left+length;
	if (right>1) right = 1;
			  
	if (variable>0)
	  outFile << " ";
	outFile << "[" << left << "," << right << "]"; 
      }
      outFile << endl;

			
      //generate branching probabilities
      BPsum=0;
      if (p.BranchingFactor==NULL){
	CurrentBF = rand()%p.MaxBranchingFactor[action];
	CurrentMaxBF=p.MaxBranchingFactor[action];
      }
      else{
	CurrentBF=p.BranchingFactor[action];
	CurrentMaxBF=p.BranchingFactor[action];
      }
			
      for(bf=0; bf<CurrentMaxBF; bf++){
	if (bf<CurrentBF){
	  BP[bf]=(double)rand()/(double)RAND_MAX;
	  BPsum+=BP[bf];
	}
	else
	  BP[bf]=0;
      }
      if (BPsum==0){
	BP[0]=1;
	BPsum=1;
      }

      for(bf=0; bf<CurrentMaxBF; bf++){
	outFile << "t ";
	//generate intervals for transition means
	for (variable=0; variable<p.Variables; variable++){
	  length = (double)rand()/(double)RAND_MAX*p.TIL[variable][action];
	  left = (double)rand()/(double)RAND_MAX*(1-length);
	  right = left+length;
	  if (right>1) right = 1;
			    
	  outFile << "[" << left << "," << right << "] "; 
	}

	//write variance
			  
	for (variable=0; variable<p.Variables; variable++){
	  length = (double)rand()/(double)RAND_MAX*(p.maxTV[variable][action]-p.minTV[variable][action]);
	  left = p.minTV[variable][action]+(double)rand()/(double)RAND_MAX*(p.maxTV[variable][action]-p.minTV[variable][action]-length);
	  right = left+length;
	  if (right>p.maxTV[variable][action]) right = p.maxTV[variable][action];
			    
	  outFile << "(" << left << "," << right << ") "; 
	}
			    	
	outFile << "p=" << BP[bf]/BPsum << endl;	

      }//end of bf (branch) loop
			
    } //end of rule loop

  } //end of action loop

  //generate rewards rules
  
  outFile << "Rewards: " << endl;
  
  for (action=0; action<p.Actions; action++){
    
    outFile << "a " << action << endl;
    //generate default
    outFile << "f d" << endl;
    outFile << "t ";
		
    outFile << "[" << 0 << "," << 1 << "] (" << p.minRV[action] << "," << p.maxRV[action] << ")" << endl;
    for (rule=0; rule<p.RewardRules[action]; rule++){
      outFile << "f";
      for (variable=0; variable<p.Variables; variable++){
	length = (double)rand()/(double)RAND_MAX*p.RIL[variable][action];
	left = (double)rand()/(double)RAND_MAX*(1-length);
	right = left+length;
	if (right>1) right = 1;
				
	outFile << " [" << left << "," << right << "]";
      }

      outFile << " and";
      for (variable=0; variable<p.Variables; variable++){
	length = (double)rand()/(double)RAND_MAX*p.RIL[variable][action];
	left = (double)rand()/(double)RAND_MAX*(1-length);
	right = left+length;
	if (right>1) right = 1;
					
	outFile << " [" << left << "," << right << "]";
      }
      outFile << endl;
      outFile << "t ";
      left=(double)rand()/(double)RAND_MAX;
      right=(double)rand()/(double)RAND_MAX;
      if (left>right){
	tmp=left;
	left=right;
	right=tmp;
      }
      outFile << "[" << left << "," << right << "] ";
			
      //generate variance interval
      length = (double)rand()/(double)RAND_MAX*(p.maxRV[action]-p.minRV[action]);
      left = p.minRV[action]+(double)rand()/(double)RAND_MAX*(p.maxRV[action]-p.minRV[action]-length);
      right = left+length;
      if (right>p.maxRV[action]) right = p.maxRV[action];
			    
      outFile << "(" << left << "," << right << ")" << endl; 

    }//end of rule loop for rewards

  }//end of action loop for rewards

  //generate termination probabilities

	
  outFile << "Termination probability:" << endl;
  outFile << "d p=" << p.minTP+(double)rand()/(double)RAND_MAX*(p.maxTP-p.minTP) << endl;
	
  for (rule=0; rule<p.TerminationRules; rule++){
    outFile << "f";
    for (variable=0; variable<p.Variables; variable++){
      length = (double)rand()/(double)RAND_MAX*p.TMIL[variable];
      left = (double)rand()/(double)RAND_MAX*(1-length);
      right = left+length;
      if (right>1) right = 1;

      outFile << " [" << left << "," << right << "]";
    }

    outFile << " p=" << p.minTP+(double)rand()/(double)RAND_MAX*(p.maxTP-p.minTP) << endl;

  }//end of rule loop for termination probabilities



  delete [] buffer;
  delete [] BP;
  delete [] outFileName;

}

void RMDP::purturb(int size, double* w, int S, int Values=0, double low=0.0, double high=1.0, int* dare=NULL)
{
  int i;
 
  for(i=0; i<size;i++){
    if (dare==NULL){
      if ((rand() % 101)>S){	
	if (Values==0)
	  w[i]=low+rand()/(double)RAND_MAX*(high-low);
	else
	  w[i]=discreteSample(low,high,Values);
      }
    }
    else{
      if (dare[i]==1){//purturb i-th value
	if ((rand() % 101)>S){	
	  if (Values==0)
	    w[i]=low+rand()/(double)RAND_MAX*(high-low);
	  else
	    w[i]=discreteSample(low,high,Values);
	}
      }
    }
  }
}  

///////


double RMDP::discreteSample(double low, double high, int V){
  int low_ind, high_ind, sample_ind;

  if (low>=1){ 
    low_ind=V-1;
  }
  else{
    
    low_ind=(int)(float)(low/(1.0/(double)(V))); 
    if (low_ind == V) low_ind=V-1;
  }
  
  if (high>=1){ 
    high_ind=V-1;
  }
  else{
    high_ind=(int)(float)(high/(1.0/(double)(V)));
    if (high_ind == V) high_ind=V-1;
  }

  sample_ind=low_ind+(rand()%(high_ind-low_ind+1));
  return ((double)(sample_ind)/(double)(V)+1.0/(double)(2*V));
  
}

////////////////////////////////////////////////
RMDPparameters::RMDPparameters(){
  rmdpName=NULL;
  UDC=NULL;
  Type='n';
  Variables=-1;
  Actions=-1;
  Discretize=NULL;
  BranchingFactor=NULL;
  MaxBranchingFactor=NULL;
  TransitionRules=NULL;
  RewardRules=NULL;
  TerminationRules=0;
  TIL=NULL;
  RIL=NULL;
  DependencyFactor=Variables;
  minTV=NULL;
  maxTV=NULL;
  minRV=NULL;
  maxRV=NULL;
  TMIL=NULL;
  minTP=0.0000001;
  maxTP=0;
  BPsimilarity=0;
  TMsimilarity=0;
  TVsimilarity=0;
  TransitionSimilarity=false;
  RMsimilarity=0;
  RVsimilarity=0;
  RewardSimilarity=false;
}

int RMDPparameters::process(int argc, char* argv[])
{
  int arg, i, j;
  int a;
  double* temp=NULL;
  char* chp;

  for(arg=0;arg<argc; arg++){

    if (strncmp(argv[arg],"?",1)==0){
      cout << "Usage: program_name parameter=value(s) parameter=value(s) ..." << endl;
      cout << "\t In the cases where the list of values has to be specified, the individual values are separated by ',' and/or '/' " << endl; 
      cout << endl;
      cout << "Parameters used by the generator (processed from command line by RMDPparameters::process(int argc, char* argv[]) method):" << endl << endl;
      cout << "rmdp \t name of the RMDP if to be saved or loaded (with path if not in the current directory)" << endl;
      cout << "T \t type of the RMDP (d for discrete and c for continuous). E.g.: T=d" << endl;
      cout << "V \t number of variables" << endl;
      cout << "A \t number of actions" << endl;
      cout << "D \t for continuous RMDPs, list the number of discrete units for piecewise-constant functions describing the RMDP dynamics (must have one number for each variable)" << endl;
      cout << "\t for discrete RMDPs, list the number of discrete values that each variable can take (must have one number for each variable)" << endl;
      cout << "\t E.g.: D=10,10" << endl;
      cout << "BF \t branching factor(s) (for each action)" << endl;
      cout << "\t E.g.: BF=5 (the same for all actions) or BF=3,5 (3-for the first action and 5 for the second action)" << endl;
      cout << "BFM \t maximum branching factor(s) (for each action), if branching factor should vary across the state space" << endl;
      cout << "\t Format as for BF" << endl;
      cout << "minTV \t lower bounds on transition variances" << endl;
      cout << "\t (one value for all variables and actions or values for each variable and action)" << endl;
      cout << "\t E.g.: minTV=0 (the same for all variables and actions) or minTV=0,0/0.01,0.01 (comma separates values for variables and '/' separates values for different actions" << endl; 
      cout << "maxTV \t upper bounds on transition variances" << endl;
      cout << "\t Format as for minTV" << endl;
      cout << "minRV \t lower bound on reward variances (one value for all actions or values for each action)" << endl;
      cout << "\t E.g.: minRV=0 (for all actions) or minRV=0.001,0.01 (values provided for each action)" << endl;
      cout << "maxRV \t upper bound on reward variances (for each action)" << endl;
      cout << "\t Format as for maxRV" << endl;
      cout << "minTP \t lower bound on termination probabilities" << endl;
      cout << "maxTP \t upper bound on termination probabilities" << endl;
      cout << "UDC \t name of the file with user defined constraints, if any (with path if not in the current directory)" << endl;

      cout << endl;
      cout << "\t Additional parameters to specify if generating random constraints:" << endl << endl;
      cout << "UDC \t in this case, name of the file to save constraints (with path if not in the current directory)" << endl;
      cout << "TR \t number of regions with distinct branching probabilities (one value for all actions or values for each action)" << endl;
      cout << "\t E.g.: TR=10 (the same for all actions) or TR=10,50 (values for each action)" << endl;
      
      cout << "RR \t number of regions with distinct reward constraints" << endl;
      cout << "\t Format as for TR" << endl;
      
      cout << "TMR \t number of regions with distinct termination probabilities. E.g.: TMR=10" << endl;
      cout << "TIL \t max interval length (between 0 and 1) for transition constraints " << endl;
      cout << "\t Format as for minTV" << endl;
      cout << "RIL \t max interval length (between 0 and 1) for reward constraints " << endl;
      cout << "\t Format as for TR" << endl;
      cout << "TMIL \t max interval length (between 0 and 1) for termination constraints (for each variable)" << endl;
      cout << "\t E.g.: TMIL=0.1 (the same for all variables) or TMIL=0.1,0.2 (values for each variable)" << endl;
      
      cout << endl;

      cout << "\t Parameters below help to generate tasks with similarities in the transition and reward structure across actions. Also they are helpfull in generating tasks with certain MDP attributes' values. Used only when generating without constraints." << endl;

      cout << endl;
      
      cout << "TS \t percentage (between 0 and 100) of similar parameters in all functions representing transition rules (branching probabilities, transition means and transition variances) for different actions" << endl;
      cout << "BPS \t percentage (between 0 and 100) of similar parameters in functions representing branching probabilities for different actions" << endl;
      cout << "TMS \t percentage (between 0 and 100) of similar parameters in functions representing transition means for different actions" << endl;
      cout << "TVS \t percentage (between 0 and 100) of similar parameters in functions representing transition variances for different actions" << endl;
      cout << "RMS \t percentage (between 0 and 100) of similar parameters in functions representing reward means for different actions" << endl;
      cout << "RVS \t percentage (between 0 and 100) of similar parameters in functions representing reward variances for different actions" << endl;
      
      return 1;
    }

    if (strncmp(argv[arg],"rmdp=",5)==0){
      rmdpName=new char[strlen(&(argv[arg][5]))+1];
      strcpy(rmdpName, &(argv[arg][5]));
    }

    if (strncmp(argv[arg],"UDC=",4)==0){
      UDC=new char[strlen(&(argv[arg][4]))+1];
      strcpy(UDC,&(argv[arg][4]));
    }

    if (strncmp(argv[arg],"T=",2)==0){
      if (argv[arg][2]=='c')
	Type='c';
      else{
	if (argv[arg][2]=='d')
	  Type='d';
	else{
	  cout << "Error: type of mdp specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }

    if (strncmp(argv[arg],"V=",2)==0){
      Variables=atoi(&(argv[arg][2]));
      DependencyFactor=Variables;
    }
    if (strncmp(argv[arg],"A=",2)==0)
      Actions=atoi(&(argv[arg][2]));
    
    if (strncmp(argv[arg],"D=",2)==0){
      if (Variables<0){
	cout << "Error: specify number of variables before D parameter" << endl;
	exit(EXIT_FAILURE);
      }
      Discretize=new int[Variables];
      a=tokenize(&(argv[arg][2]),Variables,Discretize);
      if (a==-1){
	cout << "D parameter specified incorrectly" << endl;
	exit(EXIT_FAILURE);
      }
    }



    if (strncmp(argv[arg],"BF=",3)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions before branching factors" << endl;
	exit(EXIT_FAILURE);
      }
      BranchingFactor=new int[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Actions;i++){
	  BranchingFactor[i]=-1;
	  BranchingFactor[i]=atoi(&(argv[arg][3]));
	  if (BranchingFactor[i]<0) {
	    cout << "Error: BF Specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
      }
      else{
	a=tokenize(&(argv[arg][3]),Actions,BranchingFactor);
	if (a==-1){
	  cout << "BF parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }

    if (strncmp(argv[arg],"BFM=",4)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions before branching factors" << endl;
	exit(EXIT_FAILURE);
      }
      MaxBranchingFactor=new int[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Actions;i++){
	  MaxBranchingFactor[i]=-1;
	  MaxBranchingFactor[i]=atoi(&(argv[arg][4]));
	  if (MaxBranchingFactor[i]<0) {
	    cout << "Error: BFM Specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
      }
      else{
	a=tokenize(&(argv[arg][4]),Actions,MaxBranchingFactor);
	if (a==-1){
	  cout << "BFM parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }

      
    if (strncmp(argv[arg],"DF=",3)==0)
      DependencyFactor=atoi(&(argv[arg][3]));

      

    if (strncmp(argv[arg],"TR=",3)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions before TR" << endl;
	exit(EXIT_FAILURE);
      }
      TransitionRules=new int[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Actions;i++){
	  TransitionRules[i]=-1;
	  TransitionRules[i]=atoi(&(argv[arg][3]));
	  if (TransitionRules[i]<0){
	    cout << "Error: TR parameter specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
      }
      else{
	a=tokenize(&(argv[argc][3]),Actions,TransitionRules);
	if (a==-1){
	  cout << "TR parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }

    if (strncmp(argv[arg],"RR=",3)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions before RR" << endl;
	exit(EXIT_FAILURE);
      }
      RewardRules=new int[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Actions;i++){
	  RewardRules[i]=-1;
	  RewardRules[i]=atoi(&(argv[arg][3]));
	  if (RewardRules[i]<0){
	    cout << "Error: RR parameter specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
      }
      else{
	a=tokenize(&(argv[argc][3]),Actions,RewardRules);
	if (a==-1){
	  cout << "RR parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
      }
    }

   
    if (strncmp(argv[arg],"TMR=",4)==0)
      TerminationRules=atoi(&(argv[arg][4]));

    if (strncmp(argv[arg],"TIL=",4)==0){
      if ((Actions<0) || (Variables<0)){
	cout << "Error: specify the number of actions and variables before TIL" << endl;
	exit(EXIT_FAILURE);
      }
      TIL=new double*[Variables];
      for (i=0; i<Variables;i++)
	TIL[i] = new double[Actions];

      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Variables;i++){
	  for(j=0;j<Actions;j++){
	    TIL[i][j]=-1;
	    TIL[i][j]=atof(&(argv[arg][4]));
	    if (TIL[i][j]<0){
	      cout << "Error: TIL parameter specified incorrectly" << endl;
	      exit(EXIT_FAILURE);
	    }
	  }
	}
      }
      else{
	delete [] temp;
	temp=new double[Variables*Actions];
	a=tokenize(&(argv[arg][4]),Variables*Actions,temp);
	if (a==-1){
	  cout << "TIL parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	for(i=0;i<Actions;i++)
	  for(j=0;j<Variables;j++)
	    TIL[j][i]=temp[Variables*i+j];
      }
    }

    if (strncmp(argv[arg],"RIL=",4)==0){
      if ((Actions<0) || (Variables<0)){
	cout << "Error: specify the number of actions and variables before TIL" << endl;
	exit(EXIT_FAILURE);
      }
      RIL=new double*[Variables];
      for (i=0; i<Variables;i++)
	RIL[i] = new double[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Variables;i++){
	  for(j=0;j<Actions;j++){
	    RIL[i][j]=-1;
	    RIL[i][j]=atof(&(argv[arg][4]));
	    if (RIL[i][j]<0){
	      cout << "Error: RIL parameter specified incorrectly" << endl;
	      exit(EXIT_FAILURE);
	    }
	  }
	}
      }
      else{
	delete [] temp;
	temp=new double[Variables*Actions];
	a=tokenize(&(argv[arg][4]),Variables*Actions,temp);
	if (a==-1){
	  cout << "RIL parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	for(i=0;i<Actions;i++)
	  for(j=0;j<Variables;j++)
	    RIL[j][i]=temp[Variables*i+j];
      }
    }

    if (strncmp(argv[arg],"minTV=",6)==0){
      if ((Actions<0) || (Variables<0)){
	cout << "Error: specify the number of actions and variables before minTV" << endl;
	exit(EXIT_FAILURE);
      }
      minTV=new double*[Variables];
      for (i=0; i<Variables;i++)
	minTV[i] = new double[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Variables;i++){
	  for(j=0;j<Actions;j++){
	    minTV[i][j]=-1;
	    minTV[i][j]=atof(&(argv[arg][6]));
	    if (minTV[i][j]<0){
	      cout << "Error: minTV parameter specified incorrectly" << endl;
	      exit(EXIT_FAILURE);
	    }
	  }
	}
      }
      else{
	delete [] temp;
	temp=new double[Variables*Actions];
	a=tokenize(&(argv[arg][6]),Variables*Actions,temp);
	if (a==-1){
	  cout << "minTV parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	for(i=0;i<Actions;i++)
	  for(j=0;j<Variables;j++)
	    minTV[j][i]=temp[Variables*i+j];
      }
    }

      
	
    if (strncmp(argv[arg],"maxTV=",6)==0){
      if ((Actions<0) || (Variables<0)){
	cout << "Error: specify the number of actions and variables before maxTV" << endl;
	exit(EXIT_FAILURE);
      }
      maxTV=new double*[Variables];
      for (i=0; i<Variables;i++)
	maxTV[i] = new double[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	for(i=0;i<Variables;i++){
	  for(j=0;j<Actions;j++){
	    maxTV[i][j]=-1;
	    maxTV[i][j]=atof(&(argv[arg][6]));
	    if (maxTV[i][j]<0){
	      cout << "Error: maxTV parameter specified incorrectly" << endl;
	      exit(EXIT_FAILURE);
	    }
	  }
	}
      }
      else{
	delete [] temp;
	temp=new double[Variables*Actions];
	a=tokenize(&(argv[arg][6]),Variables*Actions,temp);
	if (a==-1){
	  cout << "maxTV parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	for(i=0;i<Actions;i++)
	  for(j=0;j<Variables;j++)
	    maxTV[j][i]=temp[Variables*i+j];
      }
	  
    }

    if (strncmp(argv[arg],"minRV=",6)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions and variables before minRV" << endl;
	exit(EXIT_FAILURE);
      }
      minRV=new double[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	  
	for(j=0;j<Actions;j++){
	  minRV[j]=-1;
	  minRV[j]=atof(&(argv[arg][6]));
	  if (minRV[j]<0){
	    cout << "Error: minRV parameter specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
	  
      }
      else{
	  
	a=tokenize(&(argv[arg][6]),Actions,minRV);
	if (a==-1){
	  cout << "minRV parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	  
      }
    }
	
    if (strncmp(argv[arg],"maxRV=",6)==0){
      if (Actions<0){
	cout << "Error: specify the number of actions and variables before maxRV" << endl;
	exit(EXIT_FAILURE);
      }
      maxRV=new double[Actions];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	  
	for(j=0;j<Actions;j++){
	  maxRV[j]=-1;
	  maxRV[j]=atof(&(argv[arg][6]));
	  if (minRV[j]<0){
	    cout << "Error: maxRV parameter specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
	  
      }
      else{
	  
	a=tokenize(&(argv[arg][6]),Actions,maxRV);
	if (a==-1){
	  cout << "maxRV parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	  
      }
    }

    if (strncmp(argv[arg],"minTP=",6)==0)
      minTP=atof(&(argv[arg][6]));
	
    if (strncmp(argv[arg],"maxTP=",6)==0)
      maxTP=atof(&(argv[arg][6]));

    if (strncmp(argv[arg],"TMIL=",5)==0){
      if (Variables<0){
	cout << "Error: specify the number of variables before TMIL" << endl;
	exit(EXIT_FAILURE);
      }
      TMIL=new double[Variables];
      if ((strchr(argv[arg],',')==NULL) && (strchr(argv[arg],'/')==NULL)){
	  
	for(j=0;j<Variables;j++){
	  TMIL[j]=-1;
	  TMIL[j]=atof(&(argv[arg][5]));
	  if (TMIL[j]<0){
	    cout << "Error: TMIL parameter specified incorrectly" << endl;
	    exit(EXIT_FAILURE);
	  }
	}
	  
      }
      else{
	  
	a=tokenize(&(argv[arg][6]),Variables,TMIL);
	if (a==-1){
	  cout << "TMIL parameter specified incorrectly" << endl;
	  exit(EXIT_FAILURE);
	}
	  
      }

    }

    if (strncmp(argv[arg],"TS=",3)==0){
      TransitionSimilarity=true;
      BPsimilarity=atoi(&(argv[arg][3]));
      TMsimilarity=BPsimilarity;
      TVsimilarity=BPsimilarity;
    }

    if (strncmp(argv[arg],"BPS=",4)==0){
      TransitionSimilarity=true;
      BPsimilarity=atoi(&(argv[arg][4]));
    }

    if (strncmp(argv[arg],"TMS=",4)==0){
      TransitionSimilarity=true;
      TMsimilarity=atoi(&(argv[arg][4]));
    }
    
    if (strncmp(argv[arg],"TVS=",4)==0){
      TransitionSimilarity=true;
      TVsimilarity=atoi(&(argv[arg][4]));
    }

    if (strncmp(argv[arg],"RMS=",4)==0){
      RewardSimilarity=true;
      RMsimilarity=atoi(&(argv[arg][4]));
    }

    if (strncmp(argv[arg],"RVS=",4)==0){
      RewardSimilarity=true;
      RVsimilarity=atoi(&(argv[arg][4]));
    }

  }//end of arg loop
  
return 0;
}

void RMDPparameters::check(int task){
  //make sure all parameters are specified or substitude defaults

  
  int i, j;
  
  if ((task==generate)||(task==generate_constraints)){
    
    if (Type=='n'){
      
      cout << "Warning: type of the RMDP not specified. Using default discrete type." << endl;
      Type='d';
    }
    
    if ((Variables<0) || (Actions<0)){
      cout << "Error: Number of variables or actions not specified" << endl;
      exit(EXIT_FAILURE);
    }
      
    if (Discretize==NULL){
      
      if (Type=='c'){
	cout << "Warning: the number of discretization units not specified: using defalt 10 for each state variable " << endl;
	Discretize = new int[Variables];
	for(i=0;i<Variables; i++)
	  Discretize[i]=10;
      }
      else{
	cout << "Error: specify number of values for each variable" << endl;
	exit(EXIT_FAILURE);
      }
    }

    if ((BranchingFactor==NULL) && (MaxBranchingFactor==NULL)){
      cout << "Error: Specify branching factors" << endl;
      exit(EXIT_FAILURE);
    }

    if (Type=='d'){//make sure all transition variances are 0

      if (minTV==NULL){
	minTV=new double*[Variables];
	for (i=0; i<Variables;i++)
	  minTV[i] = new double[Actions];
      }
      for (i=0; i<Variables;i++)
	for (j=0;j<Actions;j++)
	  minTV[i][j]=0;
	
      if (maxTV==NULL){
	maxTV=new double*[Variables];
	for (i=0; i<Variables;i++)
	  maxTV[i] = new double[Actions];
      }
      for (i=0; i<Variables;i++)
	for (j=0;j<Actions;j++)
	  maxTV[i][j]=0;
    }
    else{
      if ((minTV==NULL) || (maxTV==NULL)){
	cout << "Error: specify ranges of transition variances" << endl;
	exit(EXIT_FAILURE);
      }
    }

    if ((minRV==NULL) || (maxRV==NULL)){
      cout << "Error: specify ranges of reward variances" << endl;
      exit(EXIT_FAILURE);
    }

    if (maxTP==0){
      cout << "Error: specify ranges of termination probabilities" << endl;
      exit(EXIT_FAILURE);
    }

    if (task==generate_constraints){

      if (UDC==NULL){
	cout << "Error: specify the name of the file to save constraints" << endl;
	exit(EXIT_FAILURE);
      }

      if (TransitionRules==NULL){
	TransitionRules=new int[Actions];
	for(i=0;i<Actions;i++)
	  TransitionRules[i]=0;
      }
	  
      if (RewardRules==NULL){
	RewardRules=new int[Actions];
	for(i=0;i<Actions;i++)
	  RewardRules[i]=0;
      }

      if (TIL==NULL){
	TIL = new double*[Variables];
	for(i=0;i<Variables;i++){
	  TIL[i]=new double[Actions];
	  for(j=0;j<Actions;j++)
	    TIL[i][j]=1.0;
	}
      }

      if (RIL==NULL){
	RIL = new double*[Variables];
	for(i=0;i<Variables;i++){
	  RIL[i]=new double[Actions];
	  for(j=0;j<Actions;j++)
	    RIL[i][j]=1.0;
	}
      }

      if (TMIL==NULL){
	TMIL = new double[Variables];
	for (i=0;i<Variables;i++)
	  TMIL[i]=1.0;
      }
    
    }
  }
}
  ///////////////////////
  

int RMDPparameters::tokenize(char* s, int n, int* tokens){
  char seps[]=",/";
  char* token=NULL;
  int i;
  //get first token
  token=strtok(s,seps);
  for (i=0; i<n; i++){
    if (token==NULL)
      return -1;
    tokens[i]=atoi(token);
    token = strtok( NULL, seps );
  }
  return 1;
}

int RMDPparameters::tokenize(char* string, int n, double* tokens){
  char seps[]=",/";
  char* token=NULL;
  int i;

  //get first token
  token=strtok(string,seps);
  for (i=0; i<n; i++){
    if (token==NULL)
      return -1;
    tokens[i]=atof(token);
    token = strtok( NULL, seps );
  }
  return 1;
}

int RMDPparameters::tokenize(char* string, int n, char* tokens){
  char seps[]=",/";
  char* token=NULL;
  int i;

  //get first token
  token=strtok(string,seps);
  for (i=0; i<n; i++){
    if (token==NULL)
      return -1;
    tokens[i]=token[0];
    token = strtok( NULL, seps );
  }
  return 1;
}

RMDPparameters::~RMDPparameters(){
  int i;
  delete [] rmdpName;
  delete [] UDC;
  delete [] Discretize;
  delete [] BranchingFactor;
  delete [] MaxBranchingFactor;
  if (TIL!=NULL){
    for (i=0; i<Actions; i++)
      delete [] TIL[i];
    delete [] TIL;
  }
  if (RIL!=NULL){
    for (i=0; i<Actions; i++)
      delete [] RIL[i];
    delete [] RIL;
  }
  if (minTV!=NULL){
    for (i=0; i<Actions; i++)
      delete [] minTV[i];
    delete [] minTV;
  }

  if (maxTV!=NULL){
    for (i=0; i<Actions; i++)
      delete [] maxTV[i];
    delete [] maxTV;
  }
  
  delete [] minRV;
  delete [] maxRV;
  delete [] TransitionRules;
  delete [] RewardRules;
  delete [] TMIL;


}



